// Importaciones necesarias
const express = require("express");
const dotenv = require("dotenv");
const cors = require("cors");
const helmet = require("helmet");
const xss = require("xss-clean");
const morgan = require("morgan");
const path = require("path");
const mongoSanitize = require("express-mongo-sanitize");
// connecion con la base de datos
const { sequelize } = require("./models");
// configuracion de swagger
const swaggerDocs = require("./config/swagger");
// configuracion de logger
const logger = require("./middlewares/logger.middleware");
// ruta para poder ver los logs
const logRoutes = require("./routes/logs.routes");
// ruta para los datos
const dataRoutes = require("./routes/data.routes");
// rutas para las rutas
const usuarioRoutes = require("./routes/usuario.routes");
const pacienteRoutes = require("./routes/paciente.routes");
const ubicacionRoutes = require("./routes/ubicacion.routes");
const agendamientoRoutes = require("./routes/agendamiento.routes");
const familiarRoutes = require("./routes/familiar.routes");
const coverturaRoutes = require("./routes/covertura.routes");
const atiendeRoutes = require("./routes/atiende.routes");
const autenticacionRoutes = require("./routes/autenticacion.routes");
const auditoriaRoutes = require("./routes/auditoria.routes");

dotenv.config();

// Inicialización de la aplicación
const app = express();
const PORT = process.env.PORT || 3000;

// Seguridad en encabezados HTTP
app.use(helmet());

// Sanitizar datos contra NoSQL Injection
app.use(mongoSanitize());

// Prevenir ataques XSS (Cross-Site Scripting)
app.use(xss());

// aplicando cors
const corsOptions = {
  origin: [
    "http://localhost:4200",
    "https://localhost:4201",
    "http://localhost:4202",
  ],
  // , // Solo permite este dominio
  methods: "GET,POST,PUT,DELETE,OPTIONS,PATCH,HEAD,CONNECT,TRACE,SEARCH",
  allowedHeaders: "Content-Type,Authorization",
};

app.use(cors(corsOptions));

// Middleware
app.use(express.json());

// Middleware para registrar peticiones HTTP
app.use(
  morgan(":method :url :status :res[content-length] - :response-time ms", {
    stream: {
      write: (message) => logger.info(message.trim()),
    },
  })
);

// Documentación con Swagger
swaggerDocs(app);
// Servir archivos estáticos desde la carpeta "public"
app.use(express.static(path.join(__dirname, 'public')));
// Ruta principal para servir la página
app.get('/', (req, res) => {
    res.sendFile(path.join(__dirname, 'public', 'index.html'));
});
// ruta para los logs
app.use("/logs", logRoutes);
// ruta para datos
app.use("/data", dataRoutes);
// Rutas de api
app.use("/pilar/usuario", usuarioRoutes); // tiene token
app.use("/pilar/paciente", pacienteRoutes); // tiene token
app.use("/pilar/ubicacion", ubicacionRoutes); // tiene token
app.use("/agenda/agendamiento", agendamientoRoutes); // tiene token
app.use("/pilar/familiar", familiarRoutes); // tiene token
app.use("/economia/covertura", coverturaRoutes); // tiene token
app.use("/agenda/atiende", atiendeRoutes); // tiene token
app.use("/pilar/auth", autenticacionRoutes); // tiene token
app.use("/pilar/auditoria", auditoriaRoutes); // tiene token

// Sincronización con la base de datos
sequelize
  .sync({ alter: true }) // Modifica la estructura sin eliminar datos
  .then(() => console.log("✅ Base de datos sincronizada sin perder datos"))
  .catch((err) => console.error("❌ Error al sincronizar la BD:", err));

// Levantar servidor
app.listen(PORT, () => {
  console.log(`Servidor corriendo en el puerto ${PORT}`);
});
