const express = require("express");
const usuarioController = require("../controllers/usuario.controller");
const { validatePerson } = require("../middlewares/validation.middleware");
const router = express.Router();
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware
/**
 * @swagger
 * components:
 *   schemas:
 *     Usuario:
 *       type: object
 *       required:
 *         - Nombre
 *         - Apellido
 *         - Correo
 *         - Celular
 *         - Documento
 *         - Colegiatura
 *         - FechaNacimiento
 *         - Contrasenia
 *         - CodigoVerificacion
 *         - Foto
 *         - TipoDocumentoId
 *         - GeneroId
 *         - TipoUsuarioId
 *         - EstadoId
 *         - vinculacionId
 *         - SedeId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del Usuario
 *         Nombre:
 *           type: string
 *           description: Nombre del Usuario
 *         Apellido:
 *           type: string
 *           description: Apellido del Usuario
 *         Correo:
 *           type: string
 *           description: Correo del Usuario
 *         Celular:
 *           type: string
 *           description: Celular del Usuario
 *         Documento:
 *           type: string
 *           description: Documento del Usuario
 *         Colegiatura:
 *           type: string
 *           description: Colegiatura del Usuario (solo si corresponde)
 *         FechaNacimiento:
 *           type: Date
 *           description: Fecha Nacimiento del Usuario
 *         Contrasenia:
 *           type: string
 *           description: Contrasenia del Usuario
 *         CodigoVerificacion:
 *           type: number
 *           description: Codigo Verificacion del Usuario para actualizar la contrasenia
 *         Foto:
 *           type: string
 *           description: Foto del Usuario
 *         TipoDocumentoId:
 *           type: number
 *           description: Tipo Documento del Usuario
 *         GeneroId:
 *           type: number
 *           description: Genero del Usuario
 *         TipoUsuarioId:
 *           type: number
 *           description:  Tipo Usuario del Usuario
 *         EstadoId:
 *           type: number
 *           description:  Estado del Usuario para recuperar contrasenia
 *         VinculacionId:
 *           type: number
 *           description: vinculacion del Usuario con la empresa
 *         SedeId:
 *           type: number
 *           description: Sede del Usuario
 *       example:
 *         id: "1"
 *         Nombre: "Anibal Jhamil"
 *         Apellido: "Huaman Verastein"
 *         Correo: "anibaljhamil@gmail.com"
 *         Celular: "958899251"
 *         Documento: "72547410"
 *         Colegiatura: "nop"
 *         FechaNacimiento: "1997-04-11"
 *         Contrasenia: "wasaaaaa"
 *         CodigoVerificacion: "ads"
 *         Foto: "ninguna"
 *         TipoDocumentoId: "1"
 *         GeneroId: "1"
 *         TipoUsuarioId: "1"
 *         EstadoId: "2"
 *         VinculacionId: "2"
 *         SedeId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: Usuario
 *   description: API para gestionar Usuario
 */
/**
 * @swagger
 * /pilar/usuario:
 *   get:
 *     summary: Obtiene la lista de Usuario
 *     tags: [Usuario]
 *     security:
 *       - BearerAuth: []
 *     responses:
 *       200:
 *         description: Lista de Usuario.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Usuario'
 */
router.get("/", verifyToken, usuarioController.getUsuarioAll);
/**
 * @swagger
 * /pilar/usuario/{id}:
 *   get:
 *     summary: Obtiene un Usuario por ID
 *     tags: [Usuario]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Usuario
 *     responses:
 *       200:
 *         description: Usuario encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Usuario'
 *       404:
 *         description: Usuario no encontrado.
 */
router.get("/:id", verifyToken, usuarioController.getUsuarioById);
/**
 * @swagger
 * /pilar/usuario/create:
 *   post:
 *     summary: Crea un nuevo Usuario
 *     tags: [Usuario]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Usuario'
 *     responses:
 *       201:
 *         description: Usuario creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Usuario'
 */
router.post("/create", verifyToken, validatePerson, usuarioController.createUsuario); // Aplicamos validación
/**
 * @swagger
 * /pilar/usuario/update/{id}:
 *   put:
 *     summary: Actualiza un Usuario existente
 *     tags: [Usuario]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Usuario a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Usuario'
 *     responses:
 *       200:
 *         description: Usuario actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Usuario'
 *       404:
 *         description: Usuario no encontrado.
 */
router.put("/update/:id", verifyToken, usuarioController.updateUsuario);
/**
 * @swagger
 * /pilar/usuario/{id}:
 *   delete:
 *     summary: Elimina un Usuario por ID
 *     tags: [Usuario]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Usuario a eliminar
 *     responses:
 *       200:
 *         description: Usuario eliminado exitosamente.
 *       404:
 *         description: Usuario no encontrado.
 */
router.delete("/:id", verifyToken, usuarioController.deleteUsuario);

// 🔍 Rutas individuales para búsqueda
/**
 * @swagger
 * /pilar/usuario/buscar/correo/{correo}:
 *   get:
 *     summary: Obtiene un Usuario por correo
 *     tags: [Usuario]
 *     parameters:
 *       - in: path
 *         name: correo
 *         schema:
 *           type: number
 *         required: true
 *         description: correo del Usuario
 *     responses:
 *       200:
 *         description: Usuario encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Usuario'
 *       404:
 *         description: Usuario no encontrado.
 */
router.get("/buscar/correo/:correo", verifyToken, usuarioController.buscarPorCorreo);
/**
 * @swagger
 * /pilar/usuario/buscar/celular/{celular}:
 *   get:
 *     summary: Obtiene un Usuario por celular
 *     tags: [Usuario]
 *     parameters:
 *       - in: path
 *         name: celular
 *         schema:
 *           type: number
 *         required: true
 *         description: celular del Usuario
 *     responses:
 *       200:
 *         description: Usuario encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Usuario'
 *       404:
 *         description: Usuario no encontrado.
 */
router.get("/buscar/celular/:celular", verifyToken, usuarioController.buscarPorCelular);
/**
 * @swagger
 * /pilar/usuario/buscar/documento/{documento}:
 *   get:
 *     summary: Obtiene un Usuario por documento
 *     tags: [Usuario]
 *     parameters:
 *       - in: path
 *         name: documento
 *         schema:
 *           type: number
 *         required: true
 *         description: documento del Usuario
 *     responses:
 *       200:
 *         description: Usuario encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Usuario'
 *       404:
 *         description: Usuario no encontrado.
 */
router.get("/buscar/documento/:documento", verifyToken, usuarioController.buscarPorDocumento);

// 🔍 Rutas individuales para filtrado
/**
 * @swagger
 * /pilar/usuario/filtrar/tipodocumento/{tipodocumento}:
 *   get:
 *     summary: Obtiene un Usuario por tipo documento
 *     tags: [Usuario]
 *     parameters:
 *       - in: path
 *         name: tipodocumento
 *         schema:
 *           type: number
 *         required: true
 *         description: tipo documento del Usuario
 *     responses:
 *       200:
 *         description: Usuario encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Usuario'
 *       404:
 *         description: Usuario no encontrado.
 */
router.get("/filtrar/tipodocumento/:tipodocumento", verifyToken, usuarioController.filtrarPorTipoDocumento);
/**
 * @swagger
 * /pilar/usuario/filtrar/genero/{genero}:
 *   get:
 *     summary: Obtiene un Usuario por genero
 *     tags: [Usuario]
 *     parameters:
 *       - in: path
 *         name: genero
 *         schema:
 *           type: number
 *         required: true
 *         description: genero del Usuario
 *     responses:
 *       200:
 *         description: Usuario encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Usuario'
 *       404:
 *         description: Usuario no encontrado.
 */
router.get("/filtrar/genero/:genero", verifyToken, usuarioController.filtrarPorGenero);
/**
 * @swagger
 * /pilar/usuario/filtrar/tipousuario/{tipousuario}:
 *   get:
 *     summary: Obtiene un Usuario por tipo usuario
 *     tags: [Usuario]
 *     parameters:
 *       - in: path
 *         name: tipousuario
 *         schema:
 *           type: number
 *         required: true
 *         description: tipo usuario del Usuario
 *     responses:
 *       200:
 *         description: Usuario encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Usuario'
 *       404:
 *         description: Usuario no encontrado.
 */
router.get("/filtrar/tipousuario/:tipousuario", verifyToken, usuarioController.filtrarPorTipoUsuario);
/**
 * @swagger
 * /pilar/usuario/filtrar/estado/{estado}:
 *   get:
 *     summary: Obtiene un Usuario por estado
 *     tags: [Usuario]
 *     parameters:
 *       - in: path
 *         name: estado
 *         schema:
 *           type: number
 *         required: true
 *         description: estado del Usuario
 *     responses:
 *       200:
 *         description: Usuario encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Usuario'
 *       404:
 *         description: Usuario no encontrado.
 */
router.get("/filtrar/estado/:estado", verifyToken, usuarioController.filtrarPorEstado);
/**
 * @swagger
 * /pilar/usuario/filtrar/vinculacion/{vinculacion}:
 *   get:
 *     summary: Obtiene un Usuario por vinculacion
 *     tags: [Usuario]
 *     parameters:
 *       - in: path
 *         name: vinculacion
 *         schema:
 *           type: number
 *         required: true
 *         description: vinculacion del Usuario
 *     responses:
 *       200:
 *         description: Usuario encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Usuario'
 *       404:
 *         description: Usuario no encontrado.
 */
router.get("/filtrar/vinculacion/:vinculacion", verifyToken, usuarioController.filtrarPorVinculacion);
/**
 * @swagger
 * /pilar/usuario/filtrar/sede/{sede}:
 *   get:
 *     summary: Obtiene un Usuario por sede
 *     tags: [Usuario]
 *     parameters:
 *       - in: path
 *         name: sede
 *         schema:
 *           type: number
 *         required: true
 *         description: sede del Usuario
 *     responses:
 *       200:
 *         description: Usuario encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Usuario'
 *       404:
 *         description: Usuario no encontrado.
 */
router.get("/filtrar/sede/:sede", verifyToken, usuarioController.filtrarPorSede);
/**
 * @swagger
 * /pilar/usuario/filtrar/odontologos/{sede}:
 *   get:
 *     summary: Obtiene los odontologos por sede
 *     tags: [Usuario]
 *     parameters:
 *       - in: path
 *         name: sede
 *         schema:
 *           type: number
 *         required: true
 *         description: sede del Usuario
 *     responses:
 *       200:
 *         description: Usuario encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Usuario'
 *       404:
 *         description: Usuario no encontrado.
 */
router.get("/filtrar/odontologos/:sede", verifyToken, usuarioController.filtrarPorSede);

module.exports = router;
