const express = require("express");
const ubicacionController = require("../controllers/ubicacion.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Ubicacion:
 *       type: object
 *       required:
 *         - Direccion
 *         - Ubigeo
 *         - Pais
 *         - Departamento
 *         - Provincia
 *         - Distrito
 *         - RegistroId
 *         - TipoRegistro
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del Ubicacion
 *         Direccion:
 *           type: string
 *           description: Direccion del Paciente o Usuario
 *         Referencia:
 *           type: string
 *           description: Referencia de la Direccion del Paciente o Usuario
 *         Ubigeo:
 *           type: string
 *           description: Ubigeo del Paciente o Usuario
 *         Pais:
 *           type: string
 *           description: Pais del Paciente o Usuario
 *         Departamento:
 *           type: string
 *           description: Departamento del Paciente o Usuario
 *         Provincia:
 *           type: string
 *           description: Provincia del Paciente o Usuario
 *         Distrito:
 *           type: string
 *           description: Distrito del Paciente o Usuario
 *         Latitud:
 *           type: string
 *           description: Latitud del Paciente o Usuario
 *         Longitud:
 *           type: string
 *           description: Longitud del Paciente o Usuario
 *         RegistroId:
 *           type: number
 *           description: Identifiador del Paciente o Usuario
 *         TipoRegistro:
 *           type: string
 *           description: Paciente o Usuario
 *       example:
 *         id: "1"
 *         Direccion: "Calle Cantuta 124"
 *         Referencia: "Dos cuadras antes de Unias, frente a la ferreteria Paola"
 *         Ubigeo: "243252"
 *         Pais: "Peru"
 *         Departamento: "Junin"
 *         Provincia: "Huancayo"
 *         Distrito: "Huancayo"
 *         Latitud: "-3.4324325325"
 *         Longitud: "84.4324325235325"
 *         RegistroId: "1"
 *         TipoRegistro: "paciente o usuario"
 */
/**
 * @swagger
 * tags:
 *   name: Ubicacion
 *   description: API para gestionar Ubicacion
 */
/**
 * @swagger
 * /pilar/ubicacion:
 *   get:
 *     summary: Obtiene la lista de Ubicacion
 *     tags: [Ubicacion]
 *     responses:
 *       200:
 *         description: Lista de Ubicacion.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Ubicacion'
 */
router.get("/", verifyToken, ubicacionController.getUbicacionAll);
/**
 * @swagger
 * /pilar/ubicacion/{id}:
 *   get:
 *     summary: Obtiene un Ubicacion por ID
 *     tags: [Ubicacion]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Ubicacion
 *     responses:
 *       200:
 *         description: Ubicacion encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Ubicacion'
 *       404:
 *         description: Ubicacion no encontrado.
 */
router.get("/:id", verifyToken, ubicacionController.getUbicacionById);
/**
 * @swagger
 * /pilar/ubicacion/create:
 *   post:
 *     summary: Crea un nuevo Ubicacion
 *     tags: [Ubicacion]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Ubicacion'
 *     responses:
 *       201:
 *         description: Ubicacion creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Ubicacion'
 */
router.post("/create", verifyToken, ubicacionController.createUbicacion);
/**
 * @swagger
 * /pilar/ubicacion/update/{id}:
 *   put:
 *     summary: Actualiza un Ubicacion existente
 *     tags: [Ubicacion]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Ubicacion a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Ubicacion'
 *     responses:
 *       200:
 *         description: Ubicacion actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Ubicacion'
 *       404:
 *         description: Ubicacion no encontrado.
 */
router.put("/update/:id", verifyToken, ubicacionController.updateUbicacion);
/**
 * @swagger
 * /pilar/ubicacion/{id}:
 *   delete:
 *     summary: Elimina un Ubicacion por ID
 *     tags: [Ubicacion]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Ubicacion a eliminar
 *     responses:
 *       200:
 *         description: Ubicacion eliminado exitosamente.
 *       404:
 *         description: Ubicacion no encontrado.
 */
router.delete("/:id", verifyToken, ubicacionController.deleteUbicacion);
/**
 * @swagger
 * /pilar/ubicacion/filtrar/ubicacionesde/{tipo}/{parametro}:
 *   get:
 *     summary: Obtiene un Ubicacion por usuario
 *     tags: [Ubicacion]
 *     parameters:
 *       - in: path
 *         name: tipo
 *         schema:
 *           type: string
 *         required: true
 *         description: tipo de registro de la ubicacion
 *       - in: path
 *         name: parametro
 *         schema:
 *           type: number
 *         required: true
 *         description: parametro del Ubicacion
 *     responses:
 *       200:
 *         description: Ubicacion encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Ubicacion'
 *       404:
 *         description: Ubicacion no encontrado.
 */
router.get("/filtrar/ubicacionesde/:tipo/:parametro", verifyToken, ubicacionController.filtrarPorTipoyParametro);
// router.get("/filtrar/pais/:pais", ubicacionController.filtrarPorPais);
// router.get("/filtrar/departamento/:departamento", ubicacionController.filtrarPorDepartamento);
// router.get("/filtrar/provincia/:provincia", ubicacionController.filtrarPorProvincia);
// router.get("/filtrar/distrito/:distrito", ubicacionController.filtrarPorDistrito);

module.exports = router;
