const express = require("express");
const pacienteController = require("../controllers/paciente.controller");
const { validatePaciente } = require("../middlewares/validation.middleware");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Paciente:
 *       type: object
 *       required:
 *         - Nombre
 *         - Apellido
 *         - Correo
 *         - Celular
 *         - Documento
 *         - NroHistoriaClinica
 *         - FechaNacimiento
 *         - Ocupacion
 *         - Observacion
 *         - Foto
 *         - TipoAfiliacion
 *         - FechaAfiliacion
 *         - TipoDocumentoId
 *         - GeneroId
 *         - GradoInstruccionId
 *         - SedeId
 *         - FuenteCaptacionId
 *         - OdontologoId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del Paciente
 *         Nombre:
 *           type: string
 *           description: Nombre del Paciente
 *         Apellido:
 *           type: string
 *           description: Apellido del Paciente
 *         Correo:
 *           type: string
 *           description: Correo del Paciente
 *         Celular:
 *           type: string
 *           description: Celular del Paciente
 *         Documento:
 *           type: string
 *           description: Documento del Paciente
 *         NroHistoriaClinica:
 *           type: string
 *           description: Nro Historia Clinica del Paciente
 *         FechaNacimiento:
 *           type: Date
 *           description: Fecha Nacimiento del Paciente
 *         Ocupacion:
 *           type: string
 *           description: Ocupacion del Paciente
 *         Foto:
 *           type: string
 *           description: Foto del Paciente
 *         Observacion:
 *           type: string
 *           description: Observacion del Paciente
 *         TipoAfiliacion:
 *           type: string
 *           description: Tipo Afiliacion del Paciente
 *         FechaAfiliacion:
 *           type: Date
 *           description: Fecha Afiliacion del Paciente
 *         IAFAId:
 *           type: number
 *           description: IAFA afiliado del Paciente
 *         TipoDocumentoId:
 *           type: number
 *           description: Tipo Documento del Paciente
 *         GeneroId:
 *           type: number
 *           description: Genero del Paciente
 *         GradoInstruccionId:
 *           type: number
 *           description: Grado de Instruccion del Paciente
 *         SedeId:
 *           type: number
 *           description: Sede del Paciente
 *         FuenteCaptacionId:
 *           type: number
 *           description: Fuente de Captacion del Paciente
 *         OdontologoId:
 *           type: number
 *           description: Odontologo que atendio primero al Paciente
 *       example:
 *         id: "1"
 *         Nombre: "Anibal Jhamil"
 *         Apellido: "Huaman Verastein"
 *         Correo: "anibaljhamil@gmail.com"
 *         Celular: "958899251"
 *         Documento: "72547410"
 *         NroHistoriaClinica: "72547410"
 *         FechaNacimiento: "11/04/1997"
 *         Ocupacion: "Desarrollador"
 *         Observacion: "Ninguna"
 *         Foto: "ninguna"
 *         TipoAfiliacion: "Titular"
 *         FechaAfiliacion: "11/02/2025"
 *         IAFAId: "EsSalud"
 *         TipoDocumentoId: "1"
 *         GeneroId: "1"
 *         GradoInstruccionId: "2"
 *         SedeId: "1"
 *         FuenteCaptacionId: "1"
 *         OdontologoId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: Paciente
 *   description: API para gestionar Paciente
 */
/**
 * @swagger
 * /pilar/paciente:
 *   get:
 *     summary: Obtiene la lista de Paciente
 *     tags: [Paciente]
 *     responses:
 *       200:
 *         description: Lista de Paciente.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Paciente'
 */
router.get("/", verifyToken, pacienteController.getPacienteAll);
/**
 * @swagger
 * /pilar/paciente/{id}:
 *   get:
 *     summary: Obtiene un Paciente por ID
 *     tags: [Paciente]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Paciente
 *     responses:
 *       200:
 *         description: Paciente encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Paciente'
 *       404:
 *         description: Paciente no encontrado.
 */
router.get("/:id", verifyToken, pacienteController.getPacienteById);
/**
 * @swagger
 * /pilar/paciente/create:
 *   post:
 *     summary: Crea un nuevo Paciente
 *     tags: [Paciente]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Paciente'
 *     responses:
 *       201:
 *         description: Paciente creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Paciente'
 */
router.post("/create", verifyToken, validatePaciente, pacienteController.createPaciente); // Aplicamos validación
/**
 * @swagger
 * /pilar/paciente/update/{id}:
 *   put:
 *     summary: Actualiza un Paciente existente
 *     tags: [Paciente]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Paciente a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Paciente'
 *     responses:
 *       200:
 *         description: Paciente actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Paciente'
 *       404:
 *         description: Paciente no encontrado.
 */
router.put("/update/:id", verifyToken, pacienteController.updatePaciente);
/**
 * @swagger
 * /pilar/paciente/{id}:
 *   delete:
 *     summary: Elimina un Paciente por ID
 *     tags: [Paciente]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Paciente a eliminar
 *     responses:
 *       200:
 *         description: Paciente eliminado exitosamente.
 *       404:
 *         description: Paciente no encontrado.
 */
router.delete("/:id", verifyToken, pacienteController.deletePaciente);

// 🔍 Rutas individuales para búsqueda
/**
 * @swagger
 * /pilar/paciente/buscar/correo/{id}:
 *   get:
 *     summary: Obtiene un Paciente por correo
 *     tags: [Paciente]
 *     parameters:
 *       - in: path
 *         name: correo
 *         schema:
 *           type: number
 *         required: true
 *         description: correo del Paciente
 *     responses:
 *       200:
 *         description: Paciente encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Paciente'
 *       404:
 *         description: Paciente no encontrado.
 */
router.get("/buscar/correo/:correo", verifyToken, pacienteController.buscarPorCorreo);
/**
 * @swagger
 * /pilar/paciente/buscar/celular/{celular}:
 *   get:
 *     summary: Obtiene un Paciente por celular
 *     tags: [Paciente]
 *     parameters:
 *       - in: path
 *         name: celular
 *         schema:
 *           type: number
 *         required: true
 *         description: celular del Paciente
 *     responses:
 *       200:
 *         description: Paciente encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Paciente'
 *       404:
 *         description: Paciente no encontrado.
 */
router.get("/buscar/celular/:celular", verifyToken, pacienteController.buscarPorCelular);
/**
 * @swagger
 * /pilar/paciente/buscar/documento/{documento}:
 *   get:
 *     summary: Obtiene un Paciente por documento
 *     tags: [Paciente]
 *     parameters:
 *       - in: path
 *         name: documento
 *         schema:
 *           type: number
 *         required: true
 *         description: documento del Paciente
 *     responses:
 *       200:
 *         description: Paciente encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Paciente'
 *       404:
 *         description: Paciente no encontrado.
 */
router.get("/buscar/documento/:documento", verifyToken, pacienteController.buscarPorDocumento);
/**
 * @swagger
 * /pilar/paciente/buscar/nrohistoriaclinica/{nrohistoriaclinica}:
 *   get:
 *     summary: Obtiene un Paciente por nrohistoriaclinica
 *     tags: [Paciente]
 *     parameters:
 *       - in: path
 *         name: nrohistoriaclinica
 *         schema:
 *           type: number
 *         required: true
 *         description: nro historia clinica del Paciente
 *     responses:
 *       200:
 *         description: Paciente encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Paciente'
 *       404:
 *         description: Paciente no encontrado.
 */
router.get("/buscar/nrohistoriaclinica/:nrohistoriaclinica", verifyToken, pacienteController.buscarPornroHistoriaClinica);

// 🔍 Rutas individuales para filtrado
/**
 * @swagger
 * /pilar/paciente/filtrar/sede/{sede}:
 *   get:
 *     summary: Obtiene un Paciente por sede
 *     tags: [Paciente]
 *     parameters:
 *       - in: path
 *         name: sede
 *         schema:
 *           type: number
 *         required: true
 *         description: sede del Paciente
 *     responses:
 *       200:
 *         description: Paciente encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Paciente'
 *       404:
 *         description: Paciente no encontrado.
 */
router.get("/filtrar/sede/:sede", verifyToken, pacienteController.filtrarPorSede);
/**
 * @swagger
 * /pilar/paciente/filtrar/genero/{genero}:
 *   get:
 *     summary: Obtiene un Paciente por genero
 *     tags: [Paciente]
 *     parameters:
 *       - in: path
 *         name: genero
 *         schema:
 *           type: number
 *         required: true
 *         description: genero del Paciente
 *     responses:
 *       200:
 *         description: Paciente encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Paciente'
 *       404:
 *         description: Paciente no encontrado.
 */
router.get("/filtrar/genero/:genero", verifyToken, pacienteController.filtrarPorGenero);
/**
 * @swagger
 * /pilar/paciente/filtrar/tipodocumento/{tipodocumento}:
 *   get:
 *     summary: Obtiene un Paciente por tipo documento
 *     tags: [Paciente]
 *     parameters:
 *       - in: path
 *         name: tipodocumento
 *         schema:
 *           type: number
 *         required: true
 *         description: tipo documento del Paciente
 *     responses:
 *       200:
 *         description: Paciente encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Paciente'
 *       404:
 *         description: Paciente no encontrado.
 */
router.get("/filtrar/tipodocumento/:tipodoc", verifyToken, pacienteController.filtrarPorTipoDocumento);
/**
 * @swagger
 * /pilar/paciente/filtrar/fuentecaptacion/{fuentecaptacion}:
 *   get:
 *     summary: Obtiene un Paciente por fuente captacion
 *     tags: [Paciente]
 *     parameters:
 *       - in: path
 *         name: fuentecaptacion
 *         schema:
 *           type: number
 *         required: true
 *         description: fuente captacion del Paciente
 *     responses:
 *       200:
 *         description: Paciente encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Paciente'
 *       404:
 *         description: Paciente no encontrado.
 */
router.get("/filtrar/fuentecaptacion/:fuente", verifyToken, pacienteController.filtrarPorFuenteCaptacion);
/**
 * @swagger
 * /pilar/paciente/filtrar/iafa/{iafa}:
 *   get:
 *     summary: Obtiene un Paciente por iafa
 *     tags: [Paciente]
 *     parameters:
 *       - in: path
 *         name: iafa
 *         schema:
 *           type: number
 *         required: true
 *         description: iafa del Paciente
 *     responses:
 *       200:
 *         description: Paciente encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Paciente'
 *       404:
 *         description: Paciente no encontrado.
 */
router.get("/filtrar/iafa/:iafa", verifyToken, pacienteController.filtrarPorIAFA);
/**
 * @swagger
 * /pilar/paciente/filtrar/tipoafiliacion/{tipoafiliacion}:
 *   get:
 *     summary: Obtiene un Paciente por tipo afiliacion
 *     tags: [Paciente]
 *     parameters:
 *       - in: path
 *         name: tipoafiliacion
 *         schema:
 *           type: number
 *         required: true
 *         description: tipo afiliacion del Paciente
 *     responses:
 *       200:
 *         description: Paciente encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Paciente'
 *       404:
 *         description: Paciente no encontrado.
 */
router.get("/filtrar/tipoafiliacion/:tipoafiliacion", verifyToken, pacienteController.filtrarPorTipoAfiliacion);
/**
 * @swagger
 * /pilar/paciente/filtrar/gradoinstruccion/{gradoinstruccion}:
 *   get:
 *     summary: Obtiene un Paciente por grado instruccion
 *     tags: [Paciente]
 *     parameters:
 *       - in: path
 *         name: gradoinstruccion
 *         schema:
 *           type: number
 *         required: true
 *         description: grado instruccion del Paciente
 *     responses:
 *       200:
 *         description: Paciente encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Paciente'
 *       404:
 *         description: Paciente no encontrado.
 */
router.get("/filtrar/gradoinstruccion/:gradoinstruccion", verifyToken, pacienteController.filtrarPorGradoInstruccion);
/**
 * @swagger
 * /pilar/paciente/filtrar/odontologo/{odontologo}:
 *   get:
 *     summary: Obtiene un Paciente por odontologo
 *     tags: [Paciente]
 *     parameters:
 *       - in: path
 *         name: odontologo
 *         schema:
 *           type: number
 *         required: true
 *         description: odontologo del Paciente
 *     responses:
 *       200:
 *         description: Paciente encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Paciente'
 *       404:
 *         description: Paciente no encontrado.
 */
router.get("/filtrar/odontologo/:odontologo", verifyToken, pacienteController.filtrarPorOdontologo);

module.exports = router;