const express = require("express");
const FamiliarController = require("../controllers/familiar.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Familiar:
 *       type: object
 *       required:
 *         - Nombre
 *         - RelacionId
 *         - PacienteId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del Familiar
 *         Nombre:
 *           type: string
 *           description: Nombre del Familiar
 *         RelacionId:
 *           type: string
 *           description: Relacion del Familiar con el Paciente
 *         PacienteId:
 *           type: string
 *           description: A que paciente pertenece el familiar
 *       example:
 *         id: "1"
 *         Nombre: "dsasads"
 *         RelacionId: "1"
 *         PacienteId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: Familiar
 *   description: API para gestionar Familiar
 */
/**
 * @swagger
 * /pilar/familiar:
 *   get:
 *     summary: Obtiene la lista de Familiar
 *     tags: [Familiar]
 *     responses:
 *       200:
 *         description: Lista de Familiar.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Familiar'
 */
router.get("/", verifyToken, FamiliarController.getFamiliarAll);
/**
 * @swagger
 * /pilar/familiar/{id}:
 *   get:
 *     summary: Obtiene un Familiar por ID
 *     tags: [Familiar]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Familiar
 *     responses:
 *       200:
 *         description: Familiar encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Familiar'
 *       404:
 *         description: Familiar no encontrado.
 */
router.get("/:id", verifyToken, FamiliarController.getFamiliarById);
/**
 * @swagger
 * /pilar/familiar/create:
 *   post:
 *     summary: Crea un nuevo Familiar
 *     tags: [Familiar]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Familiar'
 *     responses:
 *       201:
 *         description: Familiar creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Familiar'
 */
router.post("/create", verifyToken, FamiliarController.createFamiliar);
/**
 * @swagger
 * /pilar/familiar/update/{id}:
 *   put:
 *     summary: Actualiza un Familiar existente
 *     tags: [Familiar]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Familiar a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Familiar'
 *     responses:
 *       200:
 *         description: Familiar actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Familiar'
 *       404:
 *         description: Familiar no encontrado.
 */
router.put("/update/:id", verifyToken, FamiliarController.updateFamiliar);
/**
 * @swagger
 * /pilar/familiar/{id}:
 *   delete:
 *     summary: Elimina un Familiar por ID
 *     tags: [Familiar]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Familiar a eliminar
 *     responses:
 *       200:
 *         description: Familiar eliminado exitosamente.
 *       404:
 *         description: Familiar no encontrado.
 */
router.delete("/:id", verifyToken, FamiliarController.deleteFamiliar);
/**
 * @swagger
 * /pilar/familiar/filtro/paciente/{paciente}:
 *   get:
 *     summary: Obtiene un Familiar por paciente
 *     tags: [Familiar]
 *     parameters:
 *       - in: path
 *         name: paciente
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador del paciente
 *     responses:
 *       200:
 *         description: Familiares del paciente encontrados.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Familiar'
 *       404:
 *         description: Familiar no encontrado.
 */
// 🔍 Rutas individuales para búsqueda
router.get("/filtro/paciente/:paciente", verifyToken, FamiliarController.getFamiliarByPaciente);

module.exports = router;
