const express = require("express");
const CoverturaController = require("../controllers/covertura.controller");
const {verifyToken} = require('../middlewares/auth.middleware'); // Importar middleware

const router = express.Router();
/**
 * @swagger
 * components:
 *   schemas:
 *     Covertura:
 *       type: object
 *       required:
 *         - Nombre
 *         - CoverturaFija
 *         - CoverturaVariable
 *         - EstadoCovertura
 *         - EmpresaPaciente
 *         - IAFAId
 *         - PacienteId
 *       properties:
 *         id:
 *           type: number
 *           description: El identificador único del Covertura
 *         CoverturaFija:
 *           type: string
 *           description: Covertura Fija que cubre una aseguradora
 *         CoverturaVariable:
 *           type: string
 *           description: Covertura Variable que cubre una aseguradora
 *         EstadoCovertura:
 *           type: string
 *           description: Estado de la Covertura (activa, inactivo)
 *         EmpresaPaciente:
 *           type: string
 *           description: Empresa en la que labora el Paciente
 *         IAFAId:
 *           type: string
 *           description: IAFA relaciona la Paciente
 *         PacienteId:
 *           type: number
 *           description: Identificador del Paciente
 *       example:
 *         id: "1"
 *         CoverturaFija: "Odontopediatría"
 *         CoverturaVariable: "5%"
 *         EstadoCovertura: "Activa"
 *         EmpresaPaciente: "Clinica Dental Imagen"
 *         IAFAId: "1"
 *         PacienteId: "1"
 */
/**
 * @swagger
 * tags:
 *   name: Covertura
 *   description: API para gestionar Covertura
 */
/**
 * @swagger
 * /economia/covertura:
 *   get:
 *     summary: Obtiene la lista de Covertura
 *     tags: [Covertura]
 *     responses:
 *       200:
 *         description: Lista de Covertura.
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/Covertura'
 */
router.get("/", CoverturaController.getCoverturaAll);
/**
 * @swagger
 * /economia/covertura/{id}:
 *   get:
 *     summary: Obtiene un Covertura por ID
 *     tags: [Covertura]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Covertura
 *     responses:
 *       200:
 *         description: Covertura encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Covertura'
 *       404:
 *         description: Covertura no encontrado.
 */
router.get("/:id", verifyToken, CoverturaController.getCoverturaById);
/**
 * @swagger
 * /economia/covertura/create:
 *   post:
 *     summary: Crea un nuevo Covertura
 *     tags: [Covertura]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Covertura'
 *     responses:
 *       201:
 *         description: Covertura creado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Covertura'
 */
router.post("/create", verifyToken, CoverturaController.createCovertura);
/**
 * @swagger
 * /economia/covertura/update/{id}:
 *   put:
 *     summary: Actualiza un Covertura existente
 *     tags: [Covertura]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Covertura a actualizar
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/Covertura'
 *     responses:
 *       200:
 *         description: Covertura actualizado exitosamente.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Covertura'
 *       404:
 *         description: Covertura no encontrado.
 */
router.put("/update/:id", verifyToken, CoverturaController.updateCovertura);
/**
 * @swagger
 * /economia/covertura/{id}:
 *   delete:
 *     summary: Elimina un Covertura por ID
 *     tags: [Covertura]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: number
 *         required: true
 *         description: ID del Covertura a eliminar
 *     responses:
 *       200:
 *         description: Covertura eliminado exitosamente.
 *       404:
 *         description: Covertura no encontrado.
 */
router.delete("/:id", verifyToken, CoverturaController.deleteCovertura);
/**
 * @swagger
 * /economia/covertura/filtrar/iafa/{iafa}:
 *   get:
 *     summary: Obtiene un Covertura por iafa
 *     tags: [Covertura]
 *     parameters:
 *       - in: path
 *         name: iafa
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador de iafa
 *     responses:
 *       200:
 *         description: Covertura encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Covertura'
 *       404:
 *         description: Covertura no encontrado.
 */
// 🔍 Rutas individuales para búsqueda
router.get("/filtrar/iafa/:iafa", verifyToken, CoverturaController.getCoverturaByIAFA);
/**
 * @swagger
 * /economia/covertura/filtrar/paciente/{paciente}:
 *   get:
 *     summary: Obtiene un Covertura por paciente
 *     tags: [Covertura]
 *     parameters:
 *       - in: path
 *         name: paciente
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador del paciente
 *     responses:
 *       200:
 *         description: Covertura encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Covertura'
 *       404:
 *         description: Covertura no encontrado.
 */
router.get("/filtrar/paciente/:paciente", verifyToken, CoverturaController.getCoverturaByPaciente);
/**
 * @swagger
 * /economia/covertura/busqueda/paciente/{paciente}:
 *   get:
 *     summary: Obtiene un Covertura por ID
 *     tags: [Covertura]
 *     parameters:
 *       - in: path
 *         name: paciente
 *         schema:
 *           type: number
 *         required: true
 *         description: Identificador del Covertura
 *     responses:
 *       200:
 *         description: Covertura encontrado.
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Covertura'
 *       404:
 *         description: Covertura no encontrado.
 */
router.get("/busqueda/paciente/:paciente", verifyToken, CoverturaController.getCoverturaActivaByPaciente);

module.exports = router;