const {Usuario} = require("../models");
const bcrypt = require("bcryptjs");
const logger = require("../middlewares/logger.middleware");

// Obtener todos los usuarios
exports.getUsuarioAll = async (req, res) => {
  try {
    const usuarios = await Usuario.findAll();
    logger.info("Usuarios obtenidos correctamente");
    res.json(usuarios);
  } catch (error) {
    logger.error("Error al obtener Usuarios: " + error.message);
    res.status(500).json({ error: "Error al obtener usuarios" });
  }
};

// Obtener un usuario por ID
exports.getUsuarioById = async (req, res) => {
  try {
    const usuario = await Usuario.findByPk(req.params.id);
    if (!usuario) {
      logger.info("Usuario no encontrado por su ID");
      return res.status(404).json({ error: "Usuario no encontrado" });
    }
    logger.info("Usuario obtenido correctamente por su ID");
    res.json(usuario);
  } catch (error) {
    logger.error("Error al obtener el Usuario mediante su Id: " + error.message);
    res.status(500).json({ error: "Error al obtener usuario" });
  }
};

// Crear usuario con contraseña encriptada
exports.createUsuario = async (req, res) => {
  try {
    const { Nombre, Apellido, Correo, Celular, 
    TipoDocumentoId, Documento, Colegiatura, 
    FechaNacimiento, GeneroId, Contrasenia, 
    TipoUsuarioId, EstadoId, VinculacionId, SedeId,
    Foto, Codigo } = req.body;
    const hashedPassword = await bcrypt.hash(Contrasenia, 10);
    const usuario = await Usuario.create(
      { Nombre, Apellido, Correo, Celular, 
      TipoDocumentoId, Documento, Colegiatura, 
      FechaNacimiento, GeneroId, Contrasenia: hashedPassword, 
      TipoUsuarioId, EstadoId, VinculacionId, SedeId,
      Foto, Codigo }
    );
    logger.info("Usuarios creado correctamente");
    res.status(201).json(usuario);
  } catch (error) {
    console.log(error);
    // if (error.name === "SequelizeUniqueConstraintError") {
    //   logger.info("El correo ya está en uso");
    //   return res.status(400).json({ error: "El correo ya está en uso" });
    // }
    logger.error("Error al crear un Usuario: " + error.message);
    res.status(500).json({ error: "Error al crear usuario" });
  }
};

// Actualizar usuario
exports.updateUsuario = async (req, res) => {
  try {
    const usuario = await Usuario.findByPk(req.params.id);
    if (!usuario) {
      logger.info("Usuario no encontrado por su ID");
      return res.status(404).json({ error: "Usuario no encontrado" });
    }
    await usuario.update(req.body);
    logger.info("Usuarios actualizado correctamente");
    res.json(usuario);
  } catch (error) {
    console.log(error);
    logger.error("Error al actualizar un Usuario: " + error.message);
    res.status(500).json({ error: "Error al actualizar usuario" });
  }
};

// Actualizar solo la contraseña
exports.updatePassword = async (req, res) => {
  try {
    const { id } = req.params;
    const { contraseniaActual, nuevaContrasenia } = req.body;
    // Verificar si el usuario existe
    const usuario = await Usuario.findByPk(id);
    if (!usuario) {
      logger.info("Usuario no encontrado por su ID");
      return res.status(404).json({ error: "Usuario no encontrado" });
    }
    // Comparar la contraseña actual
    const esCorrecta = await bcrypt.compare(contraseniaActual, usuario.contrasenia);
    if (!esCorrecta) {
      logger.info("Contraseña actual incorrecta para el Usuario");
      return res.status(401).json({ error: "Contraseña actual incorrecta" });
    }
    // Encriptar la nueva contraseña
    const hashedPassword = await bcrypt.hash(nuevaContrasenia, 10);
    // Actualizar la contraseña en la base de datos
    await usuario.update({ contrasenia: hashedPassword });
    logger.info("Contraseña del Usuario actualizada correctamente");
    res.json({ mensaje: "Contraseña actualizada correctamente" });
  } catch (error) {
    logger.error("Error al actualizar solo la contrsenia del Usuario: " + error.message);
    res.status(500).json({ error: "Error al actualizar la contraseña" });
  }
};

// Eliminar usuario
exports.deleteUsuario = async (req, res) => {
  try {
    const usuario = await Usuario.findByPk(req.params.id);
    if (!usuario) {
      logger.info("Usuario no encontrado por su ID");
      return res.status(404).json({ error: "Usuario no encontrado" });
    }
    await usuario.destroy();
    logger.info("Usuarios eliminado correctamente");
    res.json({ message: "Usuario eliminado" });
  } catch (error) {
    logger.error("Error al elimininar un Usuario: " + error.message);
    res.status(500).json({ error: "Error al eliminar usuario" });
  }
};

// 🔍 Buscar usuario por correo
exports.buscarPorCorreo = async (req, res) => {
  try {
    const { correo } = req.params;
    if (!correo) {
      logger.info("Debe proporcionar un correo para buscar un Usuario");
      return res.status(400).json({ error: "Debe proporcionar un correo" });
    }
    const usuario = await Usuario.findOne({ where: { Correo: correo } });
    if (!usuario) {
      logger.info("Usuario no encontrado por su correo");
      return res.status(404).json({ error: "Usuario no encontrado" });
    }
    res.json(usuario);
    logger.info("Usuario encontrado por su correo");
  } catch (error) {
    logger.error("Error al buscar usuario por correo: " + error.message);
    res.status(500).json({ error: "Error al buscar usuario por correo" });
  }
};

// 🔍 Buscar usuario por celular
exports.buscarPorCelular = async (req, res) => {
  try {
    const { celular } = req.params;
    if (!celular) {
      logger.info("Debe proporcionar un número de celular para buscar un Usuario");
      return res.status(400).json({ error: "Debe proporcionar un número de celular" });
    }
    const usuario = await Usuario.findOne({ where: { Celular: celular } });
    if (!usuario) {
      logger.info("Usuario no encontrado por su celular");
      return res.status(404).json({ error: "Usuario no encontrado" });
    }
    logger.info("Usuario encontrado por su celular");
    res.json(usuario);
  } catch (error) {
    logger.error("Error al buscar usuario por celular: " + error.message);
    res.status(500).json({ error: "Error al buscar usuario por celular" });
  }
};

// 🔍 Buscar usuario por documento
exports.buscarPorDocumento = async (req, res) => {
  try {
    const { documento } = req.params;
    if (!documento) {
      logger.info("Debe proporcionar un documento para buscar un Usuario");
      return res.status(400).json({ error: "Debe proporcionar un documento" });
    }
    const usuario = await Usuario.findOne({ where: { Documento: documento } });
    if (!usuario) {
      logger.info("Usuario no encontrado por su documento");
      return res.status(404).json({ error: "Usuario no encontrado" });
    }
    logger.info("Usuario encontrado por su documento");
    res.json(usuario);
  } catch (error) {
    logger.error("Error al buscar usuario por documento: " + error.message);
    res.status(500).json({ error: "Error al buscar usuario por documento" });
  }
};

// 🔍 Filtrar usuarios por tipoDocumento
exports.filtrarPorTipoDocumento = async (req, res) => {
  try {
    const { tipodocumento } = req.params;
    if (!tipodocumento) {
      logger.info("Debe proporcionar un tipo de documento para filtrar usuarios");
      return res.status(400).json({ error: "Debe proporcionar un tipo de documento" });
    }
    const usuarios = await Usuario.findAll({ where: { TipoDocumentoId: tipodocumento } });
    logger.info("Usuarios filtrados por tipo de documento correctamente");
    res.json(usuarios);
  } catch (error) {
    logger.error("Error al filtrar usuarios por tipoDocumentoId: " + error.message);
    res.status(500).json({ error: "Error al filtrar usuarios por tipoDocumentoId" });
  }
};

// 🔍 Filtrar usuarios por genero
exports.filtrarPorGenero = async (req, res) => {
  try {
    const { genero } = req.params;
    if (!genero) {
      logger.info("Debe proporcionar un género para filtrar usuarios");
      return res.status(400).json({ error: "Debe proporcionar un género" });
    }
    const usuarios = await Usuario.findAll({ where: { GeneroId: genero } });
    logger.info("Usuarios filtrados por género correctamente");
    res.json(usuarios);
  } catch (error) {
    logger.error("Error al filtrar usuarios por generoId: " + error.message);
    res.status(500).json({ error: "Error al filtrar usuarios por generoId" });
  }
};

// 🔍 Filtrar usuarios por tipoUsuario
exports.filtrarPorTipoUsuario = async (req, res) => {
  try {
    const { tipoUsuario } = req.params;
    if (!tipoUsuario) {
      logger.info("Debe proporcionar un tipo de usuario para filtrar usuarios");
      return res.status(400).json({ error: "Debe proporcionar un tipo de usuario" });
    }
    const usuarios = await Usuario.findAll({ where: { TipoUsuarioId: tipoUsuario } });
    logger.info("Usuarios filtrados por tipo de usuario correctamente");
    res.json(usuarios);
  } catch (error) {
    logger.error("Error al filtrar usuarios por tipoUsuarioId: " + error.message);
    res.status(500).json({ error: "Error al filtrar usuarios por tipoUsuarioId" });
  }
};

// 🔍 Filtrar usuarios por estado
exports.filtrarPorEstado = async (req, res) => {
  try {
    const { estado } = req.params;
    if (!estado) {
      logger.info("Debe proporcionar un estado para filtrar usuarios");
      return res.status(400).json({ error: "Debe proporcionar un estado" });
    }
    const usuarios = await Usuario.findAll({ where: { EstadoId: estado } });
    logger.info("Usuarios filtrados por estado correctamente");
    res.json(usuarios);
  } catch (error) {
    logger.error("Error al filtrar usuarios por estadoId: " + error.message);
    res.status(500).json({ error: "Error al filtrar usuarios por estadoId" });
  }
};

// 🔍 Filtrar usuarios por vinculacion
exports.filtrarPorVinculacion = async (req, res) => {
  try {
    const { vinculacion } = req.params;
    if (!vinculacion) {
      logger.info("Debe proporcionar una vinculación para filtrar usuarios");
      return res.status(400).json({ error: "Debe proporcionar una vinculación" });
    }
    const usuarios = await Usuario.findAll({ where: { vinculacionId: vinculacion } });
    logger.info("Usuarios filtrados por vinculación correctamente");
    res.json(usuarios);
  } catch (error) {
    logger.error("Error al filtrar usuarios por vinculacionId: " + error.message);
    res.status(500).json({ error: "Error al filtrar usuarios por vinculacionId" });
  }
};

// 🔍 Filtrar usuarios por sedeId
exports.filtrarPorSede = async (req, res) => {
  try {
    const { sede } = req.params;
    if (!sede) {
      logger.info("Debe proporcionar una sede para filtrar usuarios");
      return res.status(400).json({ error: "Debe proporcionar una sede" });
    }
    const usuarios = await Usuario.findAll({ where: { SedeId: sede } });
    logger.info("Usuarios filtrados por sede correctamente");
    res.json(usuarios);
  } catch (error) {
    logger.error("Error al filtrar usuarios por sedeId: " + error.message);
    res.status(500).json({ error: "Error al filtrar usuarios por sedeId" });
  }
};

// 🔍 Filtrar odontologos por sedeId
exports.filtrarOdontologosPorSede = async (req, res) => {
  try {
    const { sede } = req.params;
    if (!sede) {
      logger.info("Debe proporcionar una sede para filtrar odontologos");
      return res.status(400).json({ error: "Debe proporcionar una sede" });
    }
    const usuarios = await Usuario.findAll(
      {
        where: {
          SedeId: sede,
          TipoUsuarioId: 4 // Odontologos
        }
      }
    );
    logger.info("odontologos filtrados por sede correctamente");
    res.json(usuarios);
  } catch (error) {
    logger.error("Error al filtrar odontologos por sedeId: " + error.message);
    res.status(500).json({ error: "Error al filtrar odontologos por sedeId" });
  }
};
