const {Usuario} = require("../models");
const bcrypt = require("bcryptjs");
const jwt = require("jsonwebtoken");
const dotenv = require("dotenv");
const logger = require("../middlewares/logger.middleware");

dotenv.config();
const JWT_SECRET = process.env.JWT_SECRET;

// Autenticación (Login con correo)
exports.logincorreo = async (req, res) => {
  try {
    const { correo, contrasenia } = req.body;
    // Verificar si los datos están vacíos
    if (!correo || !contrasenia) {
      logger.info("Correo y contraseña son obligatorios");
      return res.status(400).json({ error: "Correo y contraseña son obligatorios" });
    }
    // Buscar usuario por correo
    const usuario = await Usuario.findOne({ where: { correo } });
    if (!usuario) {
      logger.info("Usuario no encontrado por el correo");
      return res.status(404).json({ error: "Usuario no encontrado" });
    }
    // Verificar si la contraseña es correcta
    const esCorrecta = await bcrypt.compare(contrasenia, usuario.contrasenia);
    if (!esCorrecta) {
      logger.info("Contraseña incorrecta");
      return res.status(401).json({ error: "Contraseña incorrecta" });
    }
    const token = jwt.sign({ id: usuario.id }, JWT_SECRET, { expiresIn: "1h" });
    logger.info("Login realizado correctamente");
    res.json({ token, usuario });
  } catch (error) {
    logger.error("Error al Loguear: " + error.message);
    res.status(500).json({ error: "Error en la autenticación" });
  }
};

// Autenticación (Login con celular)
exports.logincelular = async (req, res) => {
  try {
    const { celular, contrasenia } = req.body;
    // Verificar si los datos están vacíos
    if (!celular || !contrasenia) {
      logger.info("Correo y contraseña son obligatorios");
      return res.status(400).json({ error: "Correo y contraseña son obligatorios" });
    }
    // Buscar usuario por correo
    const usuario = await Usuario.findOne({ where: { correo } });
    if (!usuario) {
      logger.info("Usuario no encontrado por el correo");
      return res.status(404).json({ error: "Usuario no encontrado" });
    }
    // Verificar si la contraseña es correcta
    const esCorrecta = await bcrypt.compare(contrasenia, usuario.contrasenia);
    if (!esCorrecta) {
      logger.info("Contraseña incorrecta");
      return res.status(401).json({ error: "Contraseña incorrecta" });
    }
    const token = jwt.sign({ id: usuario.id }, JWT_SECRET, { expiresIn: "1h" });
    logger.info("Login realizado correctamente");
    res.json({ token, usuario });
  } catch (error) {
    logger.error("Error al Loguear: " + error.message);
    res.status(500).json({ error: "Error en la autenticación" });
  }
};

// Generar un código de verificación aleatorio
exports.generarCodigoVerificacion = async (req, res) => {
  try {
    const { correo } = req.params;
    // Buscar usuario por correo
    const usuario = await Usuario.findOne(
      {
        where: {
          Correo: correo
        }
      }
    );
    if (!usuario) return res.status(404).json({ error: "Usuario no encontrado" });
    // Generar un código aleatorio de 6 dígitos
    const codigoVerificacion = crypto.randomInt(100000, 999999).toString();
    // Guardar el código en el usuario
    await usuario.update(
      {
        Codigo: codigoVerificacion,
        EstadoId: 1
      }
    );
    res.json(
      {
        mensaje: "Código de verificación generado y activado",
        codigo: codigoVerificacion
      }
      );
  } catch (error) {
    logger.error("Error al generar codigo: " + error.message);
    res.status(500).json({ error: "Error al generar el código de verificación" });
  }
};

// Verificar código
exports.verificarCodigo = async (req, res) => {
  try {
    const { correo, codigoverificacion } = req.body;
    // Buscar usuario
    const usuario = await Usuario.findOne(
      {
        where: {
          Correo: correo
        }
      }
    );
    if (!usuario) return res.status(404).json({ error: "Usuario no encontrado" });
    // Comparar código
    if (usuario.CodigoVerificacion !== codigoverificacion) {
      return res.status(400).json({ error: "Código de verificación incorrecto" });
    }
    // Marcar usuario como verificado
    await usuario.update(
      {
        Codigo: 'asdf',
        EstadoId: 2
      }
    );
    res.json({ mensaje: "Usuario verificado exitosamente" }, usuario);
  } catch (error) {
    logger.error("Error al verificar codigo: " + error.message);
    res.status(500).json({ error: "Error al verificar el código" });
  }
};

// Actualizar solo la contraseña
exports.updatePassword = async (req, res) => {
  try {
    const { id } = req.params;
    const { contraseniaActual, nuevaContrasenia } = req.body;
    // Verificar si el usuario existe
    const usuario = await Usuario.findByPk(id);
    if (!usuario) return res.status(404).json({ error: "Usuario no encontrado" });
    // Comparar la contraseña actual
    const esCorrecta = await bcrypt.compare(contraseniaActual, usuario.Contrasenia);
    if (!esCorrecta) return res.status(401).json({ error: "Contraseña actual incorrecta" });
    // Encriptar la nueva contraseña
    const hashedPassword = await bcrypt.hash(nuevaContrasenia, 10);
    // Actualizar la contraseña en la base de datos
    await usuario.update({ Contrasenia: hashedPassword });
    logger.info("Usuarios actualizado correctamente");
    res.json({ mensaje: "Contraseña actualizada correctamente" });
  } catch (error) {
    logger.error("Error al actualizar solo la contrasenia: " + error.message);
    res.status(500).json({ error: "Error al actualizar la contraseña" });
  }
};

